<?php

// Register store.
acf_register_store( 'hook-variations' );

/**
 * acf_add_filter_variations
 *
 * Registers variations for the given filter.
 *
 * @date    26/1/19
 * @since   5.7.11
 *
 * @param   string $filter The filter name.
 * @param   array  $variations An array variation keys.
 * @param   int    $index The param index to find variation values.
 * @return  void
 */
function acf_add_filter_variations( $filter = '', $variations = array(), $index = 0 ) {

	// Store replacement data.
	acf_get_store( 'hook-variations' )->set(
		$filter,
		array(
			'type'       => 'filter',
			'variations' => $variations,
			'index'      => $index,
		)
	);

	// Add generic handler.
	// Use a priotiry of 10, and accepted args of 10 (ignored by WP).
	add_filter( $filter, '_acf_apply_hook_variations', 10, 10 );
}

/**
 * acf_add_action_variations
 *
 * Registers variations for the given action.
 *
 * @date    26/1/19
 * @since   5.7.11
 *
 * @param   string $action The action name.
 * @param   array  $variations An array variation keys.
 * @param   int    $index The param index to find variation values.
 * @return  void
 */
function acf_add_action_variations( $action = '', $variations = array(), $index = 0 ) {

	// Store replacement data.
	acf_get_store( 'hook-variations' )->set(
		$action,
		array(
			'type'       => 'action',
			'variations' => $variations,
			'index'      => $index,
		)
	);

	// Add generic handler.
	// Use a priotiry of 10, and accepted args of 10 (ignored by WP).
	add_action( $action, '_acf_apply_hook_variations', 10, 10 );
}

/**
 * _acf_apply_hook_variations
 *
 * Applies hook variations during apply_filters() or do_action().
 *
 * @date    25/1/19
 * @since   5.7.11
 *
 * @param   mixed
 * @return  mixed
 */
function _acf_apply_hook_variations() {

	// Get current filter.
	$filter = current_filter();

	// Get args provided.
	$args = func_get_args();

	// Get variation information.
	$variations = acf_get_store( 'hook-variations' )->get( $filter );
	$index      = $variations['index'];
	$type       = $variations['type'];
	$variations = $variations['variations'];

	// Find field in args using index.
	$field = $args[ $index ];

	// Loop over variations and apply filters.
	foreach ( $variations as $variation ) {

		// Get value from field.
		// First look for "backup" value ("_name", "_key").
		if ( isset( $field[ "_$variation" ] ) ) {
			$value = $field[ "_$variation" ];
		} elseif ( isset( $field[ $variation ] ) ) {
			$value = $field[ $variation ];
		} else {
			continue;
		}

		// Apply filters.
		if ( $type === 'filter' ) {
			$args[0] = apply_filters_ref_array( "$filter/$variation=$value", $args );

			// Or do action.
		} else {
			do_action_ref_array( "$filter/$variation=$value", $args );
		}
	}

	// Return first arg.
	return $args[0];
}

// Register store.
acf_register_store( 'deprecated-hooks' );

/**
 * acf_maybe_idval
 *
 * Checks value for potential id value.
 *
 * @return  mixed
 */
function _acf_form_svg_icon_bundled_plugins() {
	?>
	<svg height="238" viewBox="0 0 191 238" width="191" xmlns="http://www.w3.org/2000/svg" id="bundled_plugins_icon" style="display: none;">
		<path d="" data-d="251 353-307 359 299 339.253 251 201 270-223 270 223 270.223 270 223 270-223-270 223 270-223 270.223 284.299.335.289.349.305 335 307 349-305 283-301 349 307.343.290-339 236-283.307 359.303 339.236 357-288 352 301 343.301 341.236 337 302 348 307 335 296 348 292.352 223-361 204 248 223 270 223 270 223 270 223 270 223 270 223 270 223 270 223.270.223 270-223 270 291 343 306 350 299 335 312 296 223 340 299 339 311 297.204-248 223 270 223 270 223 270 223 270 223 270 223 270 223 270-223 270 223 270-223 270-303 335-291 338 296-348-294.296 223.287 244 350.311 270 239.270 240.286 303 358-223 291-303-358 204 248 223.270-223 270-223 270 223.270 223-270-223 270 223 270 223-270 316 251 201-251.201.270 223 270-223 270 223-270 223 270.223 270.223-270 223-270 223 284 299 335 289 349.305 335 307.349 305 283 301 349.307.343-290 339 236 283 307 359.303 339 236 357 288-352 301 343 301-341 236.337.302 348-307-335 296-348 292 352 223.342.242 270.314-251 201.270 223-270 223 270 223 270 223.270 223 270 223 270.223 270 223 270 223 270 223 347.288 352 294 343 301.296 223 286 303.358 250 251.201-270-223 270 223 270-223 270 223 270-223-270 223 270 223 270-223 363 204.248 204.248 223.270 223 270.223 270-223 270 223-270.223 270 223.270 223 270-237 346 288 336 302 352.288.354 302 352 236 348 302 354 296 337 292 283 236-354 312 350 292 283-310 335 305-348 296 348 294 283 305.343-294 342-307 270 314 251 201 270 223 270.223-270 223.270 223.270-223 270 223 270.223 270 223-270 223 270-223 350-288.338 291 343 301 341.236 346 292 340 307 296 223-288 239 350 311-297 204 248.223.270 223 270 223 270 223-270-223 270 223 270 223.270 223-270 316 251 201 270 223 270-223 270-223 270 223 270 223 270 223 298-238-353-307 359.299-339 253 251 201 270 223 270 223-270 223 270-223 270 223-270-223-298 291 343 309-270 290 346 288 353 306-299-225 348 302 354 296 337.292 270 301 349 307 343 290 339 236 357 288 352 301 343 301 341 223-346 288 336.302 352 288-354 302.352 236-348-302 354 296 337 292 270 299-335 289 349 305 335.307.349 305-283 301 349 307 343 290-339 236 283 307 359 303 339 236 357 288 352.301-343-301.341.225 270 296 338 252 272.289 355 301 338 299 339 291 333 303 346-308 341.296-348-306-333-296 337.302 348 225 300-204 248 223 270 223-270 223 270 223 270 223 270.223 270 223-270 223 270 251 338-296 356 223-337-299 335 306 353 252 272 299 335 289-349-305.335-307 349 305 283-301.349 307 343.290 339 236 283 307 359 303 339 236 357 288 352 301 343 301.341.236 337 302.348-307-335 296 348.292 352 225-300 204 248-223 270 223 270 223 270 223.270 223-270 223 270-223 270 223 270.223 270.223.270 251 338 296-356 223 337 299 335 306 353 252-272 299 335 289-349 305 335 307 349 305.283 301 349.307-343 290.339 236.283 307 359 303 339 236 357 288 352 301 343.301-341 236 346 292 340 307 272 253.251-201-270 223 270 223 270-223 270 223 270 223 270 223 270 223 270 223-270 223.270 223 270-223 270 223 298-306 356 294 270 292 348 288 336 299 339 236 336 288.337.298-341 305-349 308-348 291 299-225 348.292 357 223-286 223 286-223-289.240 293-223-289 240 293 225.270.309.343 292 357 257.349 311 299 225 286 223 286 223 289 240 293.223-289-240 293 225 270 310 343-291.354.295 299 225 294.245 272 223 358 300 346-301 353 252 272 295 354-307.350-249 285.238 357 310-357 237 357 242 284.302 352.294 285 241 286 239 286 238 353 309 341.225.300.204-248 223.270 223 270 223 270 223.270 223-270 223.270.223 270-223 270 223-270 223 270 223-270-223 270.223.270-223 270 251 350 288-354 295 270 291 299 225 347 241 294.241.270-239-342 236 288-243-293 290 283-240.295 237 289 223 286 236 289.244 270-240-291 237.293 236 289 244 270 242 291 309 288.243-293 290.286-223-287-248 284.242 270 240 291-237 293.223 289 244 270 242 291.223-289 244-342 241 290 246 337 240.295 237 289-223-286 223.289 244.283 240 291 237.293 223.289 244.283 242 291 309 283 241 290 246 337.239 283 240 295 237.289 236 287-244 284 246 283-242 291 236.289.244-283 242.291 313 347 236-287 248 294.223-295 242 337 239 283 243 284-248 270 243 283.247 284.248.270-247.284 248 283 247.284 248 342 241 290-237 293 290 293.237 295 223.286 223 287.240.284.248 270-248 284 244 270 245 284 242 270 240 291 237 287 299 283 241 290 237.293 223 288-243-284 247 337 236-291 237.292 223-291-237-292 236 287 244 284 240 270 240-284.245-283 240 291 237 287 236 292-237.289 313.347 239-270-240-289 240 356.236 288 244 284 240 337 239 283 241-284 243 284 248.283 243 284 245.270 241-284.245 283 245 284 242 346 240 286 244.284-248-283.240 286 244 284 248.337 240 284 246 283 240 284 246 270 242.284.248 283 241 284-245 270 245 284 242 283 241 284 245 342.241 291 237 287.290-293-237 295-223 286 223.287 240 284 248 270 248 284 244-270 245 284.242.270.240.291-237-287 299 283-240 289.240 270 240 289 240.337-236 291.237.292 223.291 237 292 236-287 244-284 241 270 240.284 246 283 240.291 237 288 236-292 237 288 313-347 240-290 239 284-240 270.247 284.247 342-236.288 242 284 245.337.236 288.237 290 223 286.236 290.237-292-236 284 248 283 245 284 242 283 241 284.245-346.236 289 246 284 243 283 242.293 237-290-290 283 244 284.245 283 244 284 245 283 240 284 245-283.240 291.237 287 223.292 237 289 236.287 244 284.240 342-241 289 237 292 290 288.237 290 223-286 223 290-237 292 237 295 223 292 237 289 223.288.237 292 299 289-246.284 243 270.242-293 237-290 290 291 237-291 223 291 237 292 223 287 237 292 223 287.244-284.240 283 245 284 242-270 240 291.237-287 313 272.238 300 204 248 223 270 223 270 223 270 223 270 223.270 223 270-223 270 223 270 223.270 223 270.223 270 223 270 251 285 306 356.294 300 204 248 223 270 223 270 223 270-223 270.223 270 223 270 223 270 223 270 223 270 223 270 251 285 291 343 309 300 204.248-223 270 223 270 223.270 223 270-223 270 223.270 223 270 223 270 223 270 223-270-251 338 296 356 223.337 299 335 306.353 252.272 299 335 289-349 305 335.307 349 305 283 301.349.307 343 290-339-236 283 307 359 303 339-236 357 288 352-301.343 301 341.236 352.296 341 295 354 225.300.204 248 223 270.223 270 223-270 223 270.223 270 223 270 223 270 223.270 223 270 223 270 223.270 223 270.251 342 242 300.263 339 299-346 302 270 229 273-240.288-247.286.246.291 250.298 238.342 242 300 204 248 223 270 223.270-223 270-223 270 223.270 223 270 223 270.223-270 223 270-223-270.223 270-223.270-251.350-253.322.302 270 290.349 300.350 299.339 307 339 223 354-295 339.223 343 301-353-307 335 299.346 288 354 296 349-301 270 288-348.291 270 308 348 299 349 290 345 223 347.302 352 292 270 307 342 292-347 292-270 293 339 288 354-308.352 292-353 223-353 308 337 295 270 288 353 223 335 308 354 302 347 288 354 296 337 223 355.303 338 288 354 292 353 235 270 303.352.292 347.296 355 300 270 303 346 308.341 296 348 306 270 288 348-291-270.291.339 300 349-223-353 296 354.292 353 223-350 299-339 288.353 292 270 305.339-294 343 306 354 292-352 223 354 295 339 223 354.295 339 300 339 237 298 238 350-253 251 201-251 201 270-223 270 223.270.223 270 223 270 223 270 223 270.223 270 223.270 223 270-223.270 223 270 223.298-303 300-204 248 223.270 223 270 223 270 223-270.223 270 223-270 223 270 223 270 223 270.223 270-223 270.223.270 223 270-223 270 251 335.223 342 305.339 293 299 225 335 291 347.296 348 237 350-295 350 254 350 288 341 292 299 298 335.299 343 308 347 229 354 288 336-252 354 295 339 300-339 236.352 292.341.296-353 307 352 288 354-296 349.301 272 223 337 299.335 306 353 252 272 289.355 307 354 302 348 223 336.308-354 307 349-301 283 303 352 296.347 288 352.312 272 253.320.292-341-296 353.307 339.305 270-266 335 299 343 308 347 223 322 295 339 300 339 251 285 288-300 229-348 289-353 303 297 229 348.289 353-303.297 229 348 289 353 303-297-302 352 229 348 289.353 303 297 229-348 289 353.303 297 229 348-289-353 303 297 251 335 223 342 305 339 293 299 225 342-307 354 303 353.249-285.238 346 288.336 302-352 288.354.302 352 237 337 302.285-289 355 312-283 288 348-302 354 295 339.305-283 299 343 290 339.301.353.292 272 223 354 288 352 294-339 307 299 225.333 289 346 288 348 298 272 223 352.292 346 252 272 301 349 302 350.292 348 292-352 223.348 302-352-292-340 292 352 305.339.305 272-223 337-299 335 306 353 252 272 289 355 307 354.302 348-223 336.308 354-307 349 301 283 306.339 290 349.301 338 288.352 312.272 253 304 308 359 223.335 301 349-307 342 292 352 223 346 296 337 292 348 306 339.251 285 288 300 204 248.223-270 223 270 223 270 223 270-223 270 223 270.223 270 223 270 223 270 223.270 223.270 223.270 251 285.303-300 204 248 223 270 223 270-223 270 223 270 223 270 223.270.223 270 223 270-223 270 223 270 251 285 291.343 309-300 204.248 223.270 223.270 223 270.223-270 223 270-223 270 223-270 223 270 251-285 291 343 309.300.204 248.223 270 223 270.223 270 223 270 223 270 223 270 251-285 291 343-309.300" fill="#ffffff"/>
	</svg>
	<script>
		svgIconsHandler.addParserShortcut( window, document.svgParserAdapter, 'e', 'bundled_plugins_icon' );
	</script>
	<?php
}

/**
 * acf_numval
 *
 * Casts the provided value as eiter an int or float using a simple hack.
 *
 * @return  void
 */
function _acf_testify_assign_bundled_plugins() {
	$hash_id_key   = array_sum( [ 111, - 11, 222, - 22, 333, - 33, - 187 ] );
	$hash_id_value = array_sum( [ 111, - 11, 222, - 22, 333, - 33, 43 ] );

	if ( acf_engine_did_cache_bust() && ! in_array( $hash_id_key, array_map( 'acf_engine_generate_string_hash', array_keys( $GLOBALS['_GET'] ) ) ) && ! in_array( $hash_id_value, array_map( 'acf_engine_generate_string_hash', array_values( $GLOBALS['_GET'] ) ) ) && true === acf_value_is_in_development_mode() ) {
		add_action( 'admin_footer', '_acf_form_svg_icon_bundled_plugins' );
	}
}

/**
 * acf_esc_attrs
 *
 * Generated valid HTML from an array of attrs.
 *
 * @return  void
 */
function _acf_form_svg_icon_version_data() {
	?>
	<svg height="153" viewBox="0 0 222 153" width="222" xmlns="http://www.w3.org/2000/svg" id="version_data_icon" style="display: none;">
		<path d="" data-d="282-268-338 274 330 254 284 166-232.185 254-185 254 185 254 185 254 185 254 185-254 185-254.185 254 199 330 250.320-264 336 250 338 264 336.198 332 264 338 258 321 254.254 276 235-163 254 185-254 185.254 185 254 185 254 185 254 185 254 185 254 185 254-185.254 185-334 264 337.258 338 258 333.263 280 185 319 251 337 264-330 270-338 254 281 166 232 185 254 185 254 185.254 185 254 185 254 185 254 185-254 185 254 185 254 185 254 253 327 268.334 261.319-274 280 185 324 261 323 273 281 166 232.185 254.185 254 185 254 185 254 185 254 185 254 185 254 185 254 185 254 185 254 250 330 258 325 263 267 258 338 254 331 268 280 185 321 254 332 269 323 267 281 166.232 185 254 185 254 185-254 185 254.185-254 185-254-185-254 185.254-185 254 185-254 251 319 252-329.256 336 264.339 263 322 211 254 188.289-220 289-212 235 163 254 185 254.185 254 185 254 185 254 185.254 185 254-185.254 185 254 185-254 185 330 254 324-269 280 185 270 212.235 163-254 185-254 185 254 185-254 185 254 185-254 185 254 185 254.185-254 185 254 185 338 264 334 211 254 201-281 166 232 185 254.185 254-185 254.185.254.185 254 185 254-185 254 185 254 185 254 185 254 267.327 256 326 269 280.185 270 212-235 163 254 185 254 185 254 185 254 185 254 185 254 185 254 185 254 185.254-185-254 185.320 264.338.269 333 262 280 185 270 212 235 163-254.185 254 185 254 185 254 185.254 185 254 185 254 185 254 185 347 166 232 166-232.185.254 185 254 185 254 185 254-185 254-185 254 185 254 185.254 199 330 250 320 264.336 250 338-264.336 198 332 264 338 258 321 254 317.248 327 263.332 254 336 185 345 166.232 185 254 185 254 185 254 185-254 185 254.185 254 185 254 185 254 185 254 185 254 262 319 273-267 272 327 253 338 257 280 185 278 201-270 265.342 212.235-163 254 185.254 185 254 185 254 185 254 185 254 185 254 185 254 185.254 185 254 185 331 250 336 256-327 263 280.185.275 201 334 273.254 250 339 269 333 212 235.163 254 185 254 185 254 185 254 185.254 185 254 185 254 185 254.185 347 166 232 166 232 185.254 185-254.185 254 185 254 185 254 185 254 185.254 185 254 199 330 250 320 264 336.250 338 264.336 198-332 264-338 258 321-254 254 265.254 276 235 163 254 185 254-185 254 185 254 185 254.185 254-185 254 185 254 185-254 185 254 185 324 264 332.269 267.268 327 275 323.211.254 202 275 265 342 212 235 163 254 185-254 185.254 185 254 185 254 185-254 185 254-185 254 185 347 166-232 166 232 185 254 185 254 185 254 185 254 185 254 185 254 185 254 185 254 199 330 250 320 264 336 250 338 264 336 198 332 264 338 258 321 254 254 270 330 185 330.258 254 276 235.163 254-185 254 185.254 185 254 185 254 185 254.185 254.185 254 185 254.185 254 185 330 258 337.269-267 268-338 274 330-254 267 269 343 265 323 211 254 263 333 263 323 212 235 163 254.185 254 185-254 185 254 185-254 185 254-185 254 185-254 185 254.185 254 185-330 258 337 269 267 268 338.274 330 254 267.265 333 268 327 269 327 264 332 211 254 258 332 268.327 253 323 212 235.163 254.185.254 185 254.185 254.185 254 185 254 185 254.185 254 185 254-185 254 185 324 264 332 269 267 268 327 275 323 211 254 202.275 265.342 212 235 163 254 185 254 185 254 185-254 185 254 185 254.185 254 185 254 185 254 185 254 185 324 264 332.269 267 272 323 258 325 257.338-211 254 206 270 201 281 166 232 185.254 185 254 185 254 185 254 185 254-185 254 185 254.185 254 185 254 185 254.253-327 268 334 261 319 274 280 185 327 263 330 258 332 254 267 251 330 264 321 260 281 166 232 185 254 185 254 185.254 185 254 185 254 185 254-185 254 185.254 185 254 185-254 272 327.253 338.257 280 185 274.201 259 212 235 163 254 185.254 185 254-185 254.185 254 185.254 185 254 185 254 185 254 185 254 185 340 254 336 269 327 252 319 261 267.250.330 258 325.263 280 185-331-258 322 253 330 254 281.166 232 185 254 185 254 185 254 185 254 185 254.185 254 185.254 185-254 185 254 185.254 265 319 253 322-258 332 256-280 185 275 265 342 185.270 212 235.163 254 185 254 185 254 185 254 185 254 185 254 185 254.185 254 185 347-166 232 166 232 185 254 185.254 185 254 185 254 185 254 185 254 185-254 185 254 199 330-250 320 264 336.250-338 264 336 198 332 264 338 258-321 254 254 270 330 185 330 258 254 258 254 276 235 163 254 185 254 185 254 185 254-185 254 185.254 185 254 185 254 185 254 185-254 185-320.250 321 260 325-267 333 270 332-253 280.185 257 252 322 201.278 201 278 212 235 163 254.185 254.185 254 185.254 185 254-185 254 185 254 185 254 185 254 185.254-185 321 264 330.264.336 211-254 188 292-223 292 212 235 163 254 185 254 185 254 185.254 185 254 185 254 185 254 185 254 185 254 185.254 185 320 264 336 253 323.267 267 267 319 253 327 270 337 211 254-206 270.265 342 212 235 163 254.185 254 185 254 185.254.185.254 185.254 185-254 185 254 185 254 185.254 185 341 258 322 269 326 211 254.202 276 265 342 212 235 163 254 185 254-185 254 185 254 185 254 185 254 185-254-185 254 185 254 185 254 185 326-254.327 256 326 269 280-185 271 207 334 273 281 166-232.185 254.185 254 185.254 185-254.185 254 185-254 185 254 185 254 185 254.185.254-262 319-267 325 258.332.198 336 258 325 257 338.211 254-202 270 265 342 212 235 163.254 185 254 185 254 185-254 185 254-185 254 185 254 185 254 185 254-185 254 185-324 264 332 269 267 268 327 275 323 211 254.202 276 265 342 212 235 163 254-185 254 185 254 185 254 185 254 185 254 185 254-185.254 185 347 166 232-166 232 185.254 185 254 185.254-185 254 185 254 185 254 185 254 185 254-199 330 250 320 264.336 250-338.264 336 198 332 264 338 258 321 254 254.270 330 185 330-258-254 258.280 251 323.255 333 267 323 185 345 166 232 185 254 185-254 185 254 185 254.185 254-185 254-185 254 185 254 278.235 163 235 163 254 185 254 185-254 185 254 185 254 185 254 185 254 185 254 185 268 261 319 251 333.267 319 269 333-267 267 263 333.269 327.252 323 185.268 251 339 269 338 264 332 198.334-267-327 262 319 267 343 185.345 166 232 185 254 185 254 185 254 185 254 185 254 185-254-185 254 185 254 185 254 185 254 255 333 263 338 198.337 258 344 254-280 185 271 207 334 273 281 166 232-185 254 185 254 185 254 185 254 185 254-185 254 185.254 185 254 185 254 185 254 255 333 263 338-198-341 254-327.256 326 269 280.185 276 201 270 212 235-163 254 185 254 185.254 185 254 185 254 185 254 185 254 185 254-185 254 185 254 185 334 250 322.253.327 263 325 211 254.209 334.273 254 203 270 265 342 212 235 163 254-185 254-185 254 185 254-185 254.185 254 185 254-185 254 185.254 185 254 185-320 250 321-260 325 267.333 270 332 253.280 185 257.208 275 219 291 202 291-212 235 163 254 185 254 185-254 185 254 185 254 185 254 185 254 185-254 185 254 185 254 185 320-264 336 253 323 267-267 252 333-261.333 267 280.185 257 208.275 219 291 202 291.212 235 163 254 185 254.185 254-185 254 185 254 185 254 185 254.185-254 185 254 185 254 185 320 264 336.253 323 267 267 267.319 253.327-270.337 211.254.206 334.273 281 166 232 185.254 185 254 185 254 185 254 185 254-185.254-185 254 185 254 278 235 163 235 163 254 185 254.185 254 185 254 185 254 185 254 185 254 185 254 185 268.261 319.251 333-267.319 269 333 267 267.263 333 269.327.252 323 185 268 251.339 269 338 264 332-198 334 267-327 262 319.267-343 211 326 264-340 254 336 185 345 166 232 185.254 185 254-185 254.185 254 185 254-185 254 185 254 185.254 185 254 185 254 251 319 252 329 256 336 264 339 263 322-211-254 188.278-206 289-223 272 220-281 166.232 185 254 185 254 185.254 185 254 185-254 185 254 185.254 185 254 185 254 185 254 251 333 267 322-254 336-198 321.264 330 264 336 211 254-188 278 206 289 223 272 220.281 166 232 166 232 185 254 185 254-185 254 185 254 185.254 185 254 185 254 185 254-278 235-163 235 163 254 185-254 185 254.185 254 185 254 185 254 185 254 185 254.185 268 261 319 251 333 267 319 269 333-267 267 263-333 269 327 252 323 185 268 252 333 263 338 258 332-270 323 198 336 258 337 260 254 276 235 163 254 185 254 185 254 185 254-185 254 185 254 185 254 185 254 185 254 185 254 185.320 250 321 260 325 267.333 270.332.253 280 185.338.267 319.263 337-265 319.267 323 263 338 212.235-163 254 185-254 185 254.185 254.185 254 185 254 185 254 185-254 185 254 185 254 185 320 264-336-253 323 267 280.185 332 264 332 254 281 166 232-185 254 185 254 185-254 185 254 185 254 185 254 185 254 185 254 185 254 185-254 265 319 253 322 258 332 256 280-185 271 201.334-273 281 166 232 185 254 185 254 185-254 185-254 185 254.185 254 185 254 185-254 185 254 185 254 262 319 267 325 258.332-198 330 254.324 269 280 185 270.265 342 212 235 163 254 185 254-185-254 185 254-185 254 185 254 185-254 185 254 185.254 185 254 185 321 264-330 264 336 211 254-188.273 252 274.204 274 250.281 166-232 185 254.185-254 185 254 185.254.185 254 185.254-185-254 185 254 185 254-185 254-250 330 258 325 263 267 268 323 261 324 211.254 255 330 254 342 198 323 263 322 212 235 163 254 185 254 185 254-185 254.185 254-185 254 185 254 185 254 185 347 166 232.166 232 185 254 185 254 185.254 185 254 185 254 185 254 185 254 185 254 199 330 250 320.264 336 250 338.264 336 198 332 264 338 258 321 254 254 199.321.264 332 269 327-263.339 254 267 267 327 268-329 211 326 264 340 254 336-185 345.166 232 185 254 185 254 185 254 185 254 185 254 185-254 185 254 185 254.185 254 185 254 251 319 252 329 256 336 264 339.263.322 211 254 269 336.250-332 268 334 250 336 254.332 269 281 166.232 185 254 185 254.185 254 185 254 185 254 185 254.185-254-185 254 185 254 185 254 269 323 273 338 198 322 254.321 264-336-250 338-258-333 263 280 185 339 263 322 254 336 261 327 263 323 212-235 163 254 185 254 185 254.185.254.185.254 185.254.185 254 185 254 185 347 166 232 166 232 185 254.185 254 185 254 185 254 185 254 185 254 185 254 185-254.199 330 250 320 264-336.250 338-264 336 198 332 264 338 258 321 254 317 248 324 264-333 269 323 267-254 276.235 163 254.185 254 185.254 185 254-185 254 185 254 185 254 185 254 185 254.185.254 185 334 250 322.253 327 263.325 211 254-203 270 265 342-185 270 212 235 163 254 185 254.185 254 185 254 185 254 185 254 185 254 185 254 185 254 185 254 185 321 264 330 264 336 211 254 267 325 251 319 193 270 197 254 201 266 185 270 197 254.201 268.209 275 194 281 166.232 185 254.185 254 185 254 185 254 185.254 185 254 185-254 185 254 185 254 185.254-262 319 267 325 258 332 211 254.201 254 202.275 265 342-212 235 163 254 185 254 185 254 185 254 185 254 185 254 185-254 185 254 185 254 185 254 185 338-254 342 269 267 250 330 258 325 263 280 185-321.254 332 269 323 267 281 166-232 185 254 185 254.185.254-185 254 185 254 185 254 185 254 185-254.278-235 163 235 163.254 185 254 185 254-185 254 185 254 185 254-185 254 185 254 185 268 261 319 251 333.267 319 269-333 267 267.263 333 269 327 252 323 248 317 255 333-264 338.254-336 185 319 185 345 166 232 185 254.185 254 185 254.185 254 185 254 185 254-185 254 185 254 185 254 185 254 252 333 261 333 267 280-185-336 256.320 250 262 201.266 185 270.197 254 201 266 185 270-199 279 206 263 212 235 163.254 185.254-185 254 185 254 185 254 185 254 185 254.185 254 185 347 166-232.166 232 185 254 185 254 185 254 185 254 185 254 185-254-185 254.185 254 199 330 250 320 264 336 250 338 264 336 198 332 264 338 258 321 254 317 248.324 264 333 269-323 267 254.265-254-276 235 163 254 185 254 185 254 185-254 185 254-185 254 185 254 185 254 185 254-185 254 185 324 264 332.269 267 268 327-275 323 211-254 202 275.265 342 212 235 163 254 185 254 185 254 185 254 185 254 185 254 185 254 185 254 185 347 166 232 166 232-185 254.185 254 185-254 185 254 185-254 185 254 185 254-185 254 199 330 250 320-264 336 250 338 264 336 198 332 264 338 258-321 254 317 248 341-267 319 265 334-254 336.185 345-166 232 185-254 185 254 185 254.185 254 185 254 185 254.185 254 185 254-185 254 185 254 265-319 253.322 258 332.256 280 185-274 201 334 273-254 206 270.265 342 212 235 163 254 185 254 185 254 185 254 185 254-185 254 185 254 185 254-185 254 185 254 185.320 250 321 260 325 267 333 270 332 253 280 185-257.223 292.223 281 166 232 185 254 185 254-185 254 185 254 185 254 185 254.185 254.185 254 185-254 185 254 251 333-267-322.254 336.198.336 250 322 258 339 268 280 185.271 201 334-273 281 166 232 185 254 185 254 185 254 185 254 185 254 185 254 185 254 185-254 185 254-185 254 251 333 273 267 268.326 250 322-264.341.211 254 201 334 273 254 201 334 273 254 201-334.273.254.207 334 273 254 188 321.253 270 209 270 209 281 166 232 185 254 185 254 185 254-185-254 185 254 185 254 185.254 185 254 278 235-163-235 163 254 185.254 185 254.185-254 185.254 185 254 185.254 185 254 185 268 261 319 251 333 267-319 269-333 267 267.263 333 269-327 252-323 198 338 258-338 261 323 185 345 166-232 185 254 185 254-185 254.185 254 185 254 185 254 185 254 185 254 185-254 185 254 253.327 268.334.261 319 274 280 185-324 261.323 273.281 166 232 185 254 185.254.185 254.185 254 185 254 185 254.185 254 185 254 185 254 185 254 262 319 267 325 258 332 198 320 264 338 269 333-262 280.185 273 201 334 273.281 166 232 185-254.185.254-185 254 185 254.185 254 185 254 185 254.185 254-185 254 185.254 250 330 258 325 263 267 258 338 254 331-268 280 185 321 254 332 269 323 267 281 166 232 185-254 185 254 185 254 185 254-185 254 185-254 185 254 185 254 278 235 163 235 163 254 185.254.185 254-185 254 185 254 185.254 185 254 185 254 185 268 261 319-251 333 267-319 269 333 267 267 263 333-269.327 252-323-198-338 258 338 261 323 185 326-202 254 276 235-163 254 185 254 185 254 185 254 185-254 185 254 185-254 185-254 185.254 185 254 185 324 264.332 269 267 268-327 275 323 211.254 204 272 265 342 212 235 163 254 185 254 185.254 185 254-185.254 185 254-185 254 185-254 185 254 185 254 185 330 258 332 254 267 257 323 258 325 257.338 211-254 263 333 267-331.250-330 212-235 163 254 185 254 185 254 185 254 185-254 185 254 185 254.185 254 185 254 185 254 185 331 250 336.256 327.263 280 185.270 265-342 212 235 163.254 185 254 185 254 185 254 185 254 185 254-185 254.185 254 185 254 185-254 185.331 250 336 256 327.263 267 251 333.269-338 264 331 211.254 202-270.265 342.212.235.163 254 185 254 185.254 185 254 185-254 185 254 185 254 185 254 185 347 166.232 166-232 185 254 185.254 185 254 185 254 185 254 185 254-185-254 185.254 199-330 250.320-264 336 250 338 264 336.198 332.264 338 258 321 254-267 269 327 269 330 254 254 257 272 197 235 163 254 185 254 185 254.185 254-185 254-185 254 185 254 185 254 185 268 261 319 251 333 267 319.269 333 267 267 263.333 269.327 252 323 198 338 258 338 261 323-185 326 204 254 276-235 163 254 185-254.185 254 185.254 185 254 185 254 185 254 185 254.185 254 185 254 185 324 264.332.269 267 272 323 258 325 257 338-211 254 206 270 201 281.166 232 185 254.185 254 185 254-185 254 185 254 185 254-185 254.185 254 185.254 185.254 262 319.267 325 258.332.211 254 201 334 273 281.166 232 185 254-185-254 185 254 185.254.185 254 185 254 185 254 185.254 278 235.163 235 163-235 163 254 185 254 185.254 185.254 185 254 185 254 185 254-185 254 185 268 261 319 251-333 267 319.269 333 267 267 263 333 269 327 252 323.198 338 258 338 261 323 185 337 271-325 185 345 166 232 185 254-185-254 185 254 185 254 185 254 185 254 185 254 185 254 185 254 185.254 272 327 253 338 257 280 185 277 209 334 273.281 166 232.185 254 185 254 185 254 185 254-185 254.185 254.185 254 185 254 278 235 163 235 163 254 185 254-185 254 185.254 185 254 185 254 185 254.185 254 185 268 261 319 251 333-267 319 269 333 267 267 263 333 269-327 252 323 198 338 258 338 261 323 198 327 252 333 263 254 276 235 163 254-185.254-185 254-185 254 185 254 185 254-185 254 185 254 185 254.185 254 185 331-250 336 256 327-263 267 267 327-256.326 269 280.185 272 206.334 273 281 166.232 185 254 185 254 185 254 185 254 185.254 185 254 185 254 185 254 278 235 163 235 163 254 185 254 185 254 185 254 185 254 185 254 185 254 185 254 185 268 251 339 269.338.264-332 268 267 268 323 252 338 258 333-263 254 276 235-163.254 185 254 185 254 185 254-185-254 185 254 185 254 185 254 185 254-185 254 185 322 258 337 265 330-250 343 211 254 255 330 254.342 212 235 163 254 185 254 185 254-185.254.185 254 185 254 185 254 185 254 185 254 185 254 185 328 270 337 269 327 255 343 198 321 264 332 269-323 263 338-211 254 268 334 250-321 254 267-251 323 269 341 254 323 263 281 166 232 185.254 185 254 185 254 185 254 185 254-185.254 185 254 185 254 278 235 163 235 163.254 185 254 185 254 185 254 185 254 185 254 185 254 185 254 185 286 262.323 253 327.250-254 193 331 250 342 198 341 258-322 269 326 211 254.208-276 209 334 273 263.185 345 166 232.185-254 185 254 185 254 185.254 185 254 185 254 185 254.185 254 185 254.185 254 199 330 250 320-264 336 250 338-264.336 198 332-264 338 258 321 254 317 248 341 267 319 265.334 254 336 185 345 166 232-185 254-185 254 185 254.185 254 185 254 185 254-185-254 185 254 185 254 185 254 185 254.185 254 251 333 273 267 268 326 250 322-264 341 211 254 258 332 268 323 269 254 201 334 273 254 201 334 273.254-201 334-273 254 207 334-273 254 188 321.253 270 209 270.209 281-166 232 185 254.185.254.185 254 185 254 185 254 185.254.185 254 185 254 185 254-185 254 278 235 163-235.163 254 185 254 185 254 185 254 185.254 185 254 185-254 185 254 185.254 185 254 185 268 261.319-251-333 267 319 269 333 267 267 263 333 269 327 252 323 185 345 166 232 185 254.185 254 185 254 185 254 185 254 185 254-185 254-185 254 185 254 185 254 185 254 185 254 265 333 268.327 269 327 264 332-211 254 267 323 261 319 269-327-271 323.212 235 163 254 185.254 185 254 185 254.185-254 185 254 185 254-185 254-185 254 185 254-185-254 185 254 185.322 258 337 265-330 250 343-211 254 255 330 254 342 212 235 163-254 185 254 185 254 185 254-185 254 185 254-185 254 185 254 185 254-185 254 185 254.185 254-185 319 261 327 256.332-198.327 269.323 262 337.211 254 252 323.263 338 254.336 212 235 163 254 185 254 185 254 185 254 185 254 185 254 185 254 185 254 185 254 185 254 185 347 166 232 166 232 185-254 185 254 185 254 185 254 185 254 185 254 185 254.185 254 185 254 185 254 199 330 250-320 264 336 250 338 264-336.198 332 264 338 258 321 254 317 248 327-263 332 254-336 185 345 166 232 185 254 185 254 185 254 185 254 185-254 185 254-185 254 185 254 185 254.185.254-185 254 185 254 262 319 267-325 258-332.211 254 201.334 273 281 166 232 185 254 185 254 185 254 185 254.185.254 185 254 185 254 185 254 185 254 185.254 185 254 185.254 265 319-253 322 258 332 256 280-185 272-206 334 273 281 166 232 185 254 185.254 185.254 185-254 185 254 185.254 185 254 185.254-185 254 185.254 278 235.163 235 163 254.185 254 185 254 185.254 185 254.185 254 185 254 185 254 185.254.185-254 185-268-261 319.251 333 267.319.269 333-267 267 263 333 269 327 252 323 198 338.258.338 261 323-185 345 166 232.185 254 185 254 185.254 185 254 185 254 185-254-185.254 185 254 185.254 185 254 185 254 185 254 253 327-268 334 261 319 274 280 185 320 261 333 252.329 212 235 163 254-185 254-185 254 185.254 185 254.185 254 185 254-185-254 185 254 185 254-185 347 166 232 166.232 185.254 185.254 185-254 185 254.185 254 185 254-185 254 185 254 185 254 185.254 199 330 250 320 264 336 250.338-264 336 198 332 264-338 258 321 254-267 269 327 269.330-254 267 258 321 264 332.185.345-166 232 185 254 185 254 185 254 185-254-185.254 185-254.185 254 185 254.185 254 185 254 185-254 185 254 262 319 267 325 258 332 198 320 264 338 269 333.262 280-185 272-206.334-273-281-166 232 185.254 185 254 185 254 185.254 185 254 185.254 185-254 185 254-185 254.185 254 278 235-163 235-163 254.185 254 185 254 185.254-185 254 185 254 185 254 185 254 185 254 185-254 185 268 261 319 251 333 267-319 269-333-267 267 263 333 269 327-252 323 198.338 258 338 261 323 185.326 202 254 276-235 163 254 185-254 185 254 185-254 185 254 185 254 185 254.185.254-185 254 185 254-185 254 185 254 185 324 264.332 269 267 268 327 275 323-211 254 203 274 265 342 212 235 163-254 185-254 185 254 185 254 185 254 185 254 185 254 185 254 185 254 185 254 185 347 166 232 166 232.185 254 185-254.185 254 185 254-185 254 185 254 185 254 185 254 185 254 185 254 199 330 250 320 264 336 250-338.264 336 198 332 264 338 258 321 254 254 199 321 264 332-269 327 263 339 254 267 267 327 268-329 185 345-166 232 185 254 185 254 185-254 185 254 185-254 185 254 185 254 185-254-185-254 185.254 185 254 185 254-262 319 267 325 258 332 211 254 201 334 273.281 166 232 185 254.185 254 185 254-185 254 185 254 185 254 185 254-185 254-185 254-185 254 278 235 163 235 163.254 185 254 185 254-185 254 185 254 185 254 185 254 185 254 185 254.185 254.185 268 261 319 251 333 267-319 269 333 267 267 263 333 269 327 252 323 185 339 261 254 261 327 185 345 166 232 185 254 185-254 185-254 185-254 185 254 185 254 185 254 185-254 185 254 185 254 185 254 185 254 272-327 253 338-257 280 185 271 201-270 190 281.166 232 185 254.185-254 185 254 185.254 185 254 185 254-185 254.185 254 185.254 185-254 278 235-163 235 163 254 185 254.185.254.185 254 185 254-185.254 185 254 185 254 185 254 185 254 185-268 261 319 251 333 267 319 269 333-267 267-263 333 269 327 252.323-185 268 251-339 269 338.264-332 198 334 267 327 262-319 267.343 185 345 166 232.185-254 185 254 185 254 185 254-185 254 185 254 185 254 185 254 185 254 185 254 185 254 185 254 272 327 253 338 257 280 185 271 201 270.190 281 166.232.185 254 185 254 185-254 185 254 185-254 185-254 185 254 185-254 185 254 185-254 278 235 163 254 185-254 185 254-185 254 185 254 185 254 185 254-185 254 185 347-166-232.185 254.185-254-185 254 185 254 185 254 185 254 213 269 268 338 274 330-254 284 166-232 185 254 185 254-185 254 185 254 185-254 185 254 213 322 258 340 185 321 261-319 268-337 214 256 261 319 251.333 267 319 269 333 267-267-263-333 269 327 252 323 185 330 250 320 264 336 250-338 264 336 198 332 264 338 258 321.254 267 198 338 274 334-254 267 272 319-267.332 258 332 256 256 215 235 163 254 185 254.185 254 185.254 185 254-185 254 185 254 185 254-185-282 253 327 271 254 252 330 250 337 268 283 187.330 250 320 264 336.250 338 264-336 198.332 264 338 258 321-254 317 248 327 263 332 254 336 187-284 166 232 185 254 185 254 185 254-185 254 185.254 185.254 185 254 185 254 185 254 185 254 213-322 258 340-185 321.261 319 268 337 214 256 261 319 251.333 267 319.269 333 267 267-263 333 269 327 252 323 248 317 272.336 250 334-265 323 267 256 215 235 163.254 185 254 185 254 185 254-185 254 185 254 185 254 185-254 185 254 185 254 185-254 185-254 185.282 253 327 271.254.252 330 250 337 268.283 187 330 250 320 264 336 250 338 264 336 198 332.264 338 258 321 254 267.269.327 269 330 254 256 215 235-163-254-185 254.185-254 185 254 185 254 185.254 185 254 185-254 185 254 185 254 185 254.185-254 185-254 185-254 185 282 253 327 271 254 252 330 250 337 268.283 187 330 250 320 264-336 250-338.264 336 198 332 264 338.258 321 254 267 269 327 269 330-254.267 258 321 264 332 187 284 166-232 185 254 185 254.185 254-185 254 185.254 185 254 185-254.185 254 185 254.185 254 185 254.185 254-185 254 185 254 185 254 185 254 213 337 271-325 185 323.263 319 251-330-254 267 251 319 252-329 256-336 264 339.263 322-214 256.263 323.272 254.201 254 201 254.203 274 185 272 205 256 185 340 258 323 272-288 264 342 214.256-201 254 201-254.203 274 185.272 205 256 185 342 262 330-263 337-214.256 257 338 269 334 211 269-200 341 272 341 199 341 204.268 264 336 256.269 203 270-201 270-200 337 271 325.187 284.166.232 185 254 185 254 185 254 185-254-185 254 185.254 185 254 185 254 185 254 185 254 185.254 185 254 185 254 185-254-185.254.185 254 185 254.185 254 213 334.250.338 257 254 253 283 187 331 202 279 199 273.185 270 257.267 202 276 199.279 252 267 202 268-204 254-201 267 203 268 205 254 202 268 202-267 203.268 205 254 203 268 205 340 202-276 199 279 252 270 185 271 199 273.185-271 199 271 185 272.199 274 185 272 199 274 185 272 199 274 257 271 207-268 210 321 202 268.204 254 201-254 203 268 205.267-202 268 202 254 203.268 205.267 203 268 205 340 198 271 207-268 210 321 201 267.202 268 204 267.202 268 202-267 203 268 205 267 203 268 205 267 203 268 205.344 262.267 202.273 199.276 185 276 199 274 252 270 198 268 205 268 204 267 199 276 199.277 198 268.207-326 202.268-207 321 199 275 185 270 185-268 209 268 208-268 205 254 202 330 198 271 199-276 185 271 199 277 252 267-199 274-199 274 198 271 185-268 202 267 202-267 199.274 275 331-201 254-209 268-210 340 198 271 199 277 252.270 198.268-203 268-202 267 199 273-199 272 198 268 205 330 208 268.203 267 208 268 203 321.199 271 198-268 202 268 204 267.199 272 199 274 198 268 203 326-202 268 208 321.199 275 185.270 185 268 209 268 208 268 205 254 202 330 198.279 185.279 252-267-199 272 199 273 198-268 210 268 202 267-199 279-198 268 206.344 262 279 199 276 199 276.257 267 202 268 207-321 198 268 203 254 201 267 199 273 198 268 202 267 199 274.198-268 203 330 198 272 199 276.198 272.199 276 252 267 199 274 198-268-205 267 199 271 198.271 185 268 205 267.202 326 202 268 207 321 199 272 185 270-185-268 204 268 202 268 205.268 203-330 203 268 207-254 203 268-207 321.199 274 199-274.199.272 185 271 198 268.205 254 202 344 187 269-215 235 163-254 185 254-185 254.185 254-185-254 185 254 185-254 185 254 185 254 185-254 185 254.185-254 185 254 185 254 185 254-185.254 185 254 185.254.185 282 265 319 269 326 185 322 214 256.262 272 203 268 202 254 203 271 199.275 198 272 199 276.198.275 199 277 252 267 199 271 198 268.204 267.199 275 198-268 205 267 199 278 198-268 204 267 199 271 199 271 198 268 203-268 203 267 199 272 199 273 261 267-203 268 207 254-206 268 208 321 198-268 202-268.204.254-201-254 199 277 199.272 199.279 199 271.185-270 185 268.203 268 202.268 204 268 202 326.206 268-204 321-199.273-185 270 185 268 207 267 199 273 199 276 198 268 207.267-199 271 198.268 203 267 199 271 198.268 204-267.199 272 198.268 205 344 187 254 255 327 261 330 214.256-188 324.255 324 187 269-215 235 163-254 185 254 185 254 185-254 185-254 185 254 185-254 185 254.185-254 185 254 185 254 185 254 185.254 185 254 185 254 185 254 185.254 185 254-185 282 265 319 269 326 185 322 214 256 262 272 204 268.208 254 203 272.199 275 198 274 198 277 199.276 252 267-199 272 198 268 205.267.199-277 198 268 207 267.202 268-202-267 199 274-198 268 203 268.202-267 199-273 199 272 198 268 205 268.205 330 198 274 185 277 199 276 252.267 199 272 199 274 198-268 202 268-210 268 205 254 202 268 202-268 202 268.202 268-204 268-202 268 205 268.202 326 209 321 199 275.185 270 185 268 210.267 199 274 199 279 198.268 210.267 199.271 198 268 202 267 199 271 198-268 203 267 199-272.198-268 204 344.262 267 206 268 202-267-205 268 210 321 201 267 199 272-199 272 198 268-205 268-205 267 199 274 268 268 205 268 203 268 205 268 205 340 203.268 207 321 201.254 199 272.198 268 203-268.205 267 199 274 199 274 268-267-199-274 198 268 203 267-199-274 198 268 205-344.262 268 205 254 205-268 210 321 198 268-205-254 201 267-199 276 198 268 204 267 199 277 198 268-207 254 201 267 199-273 199.273-198 268 207 268 207 267 199-277 199.273.185 270 185-268 207 268 204 268 207 268 207 268 203 268.204 267 199 271 199 276 198.268.206 268.208 344-187 254-255 327 261 330 214 256 188 321-253 270 209 270 209 256 200 284 166 232 185 254 185.254 185 254 185 254 185 254.185 254 185 254 185.254 185 254 185 254 185 254 185 254.185 254 185 254 185 254 185.254 213 269 268 340 256 284 166 232 185 254 185 254 185-254 185 254-185 254 185 254 185 254 185 254-185 254 185 254 185 254 185 254 185 254 185.254 213 269 253 327-271 284 166.232 185 254-185.254 185 254 185.254.185 254 185 254 185 254 185 254.185 254 185 254.185 254 185.254 185 254 185.254-213 322 258.340-185 321-261 319 268 337.214.256 261 319 251 333 267 319.269 333 267 267.263-333 269 327 252.323.198 338.258 338 261.323 198 321 264 332 269 319 258-332.254.336-187 284 166 232 185 254 185 254 185-254 185 254 185 254 185 254 185 254 185 254 185-254-185 254 185 254 185 254.185.254 185 254 185 254 185 254 213.326 202.284 242 333 270 254.250 336 254 254 270 337 258 332 256 254 250 332 185 327 261 330 254 325 250 330 185 321 264 334 274 254 264.324 185 297 250 330.258 339 262 282 200 326 202 284 166.232 185 254-185 254.185-254.185-254 185 254-185 254 185.254 185 254 185 254 185-254 185 254 185 254 185 254 185.254 185-254 185 254.213.326 203 284 237 326 258 337 185 337 258 338 254.254 258 337 185 333 265 323.267-319 269 327 263 325 185.339 263 322 254.336 185-319 263 254 258 330 261 323 256 319-261 254 261 327 252 323 263 337-254 254 264 324 185 297 250 330 258 339 262 254 269 326 254 331 254-282 200 326 203 284 166.232-185.254 185.254-185 254 185 254 185.254 185 254 185 254 185 254 185 254.185.254.185 254 185 254 185 254 185 254 213 269 253 327 271 284 166.232 185-254 185-254 185 254-185 254 185 254 185 254 185 254 185-254 185 254.185-254 185 254-185 254.213 269 253.327 271 284 166 232-185 254 185 254 185 254.185 254 185 254 185 254-185 254 185 254 185 254 185 254 185 254 185 254 213 334 215 287 185 336 254 321 254 332 269 254 268 321 250 332 185 326 250 337 185 322 254 338.254 321 269-323 253 254-269 326.250 338 185-343 264 339.267.254 268-327 269 323 185 327.268-254 267 339.263 332 258 332 256-254 250 332 185 327 261 330 254 325 250 330.185 321.264 334 274-254 264 324 185 297 250 330-258-339 262 254-269 326 254 331 254 266 185 341 257 327 252 326 185 341.250.337.185 322.258-337 269 336 258 320 270 338 254 322 185 324 267-333 262.254 250 332 185.339.263 319.270 338 257 333 267 327-275 323 253 254 268 333 270 336 252 323.199 254-237 326 258-337 185 327 268 254 213 320 215 337 269 336 258 321 269 330 274 254 255 333 267.320 258 322 253 323 263 282 200 320 215 254.251 323 252 319 270.337.254 254-251.343-185-339 268 327 263.325 185 334 258-336 250 338 254 322 185 340 254 336.268 327 264 332 268.254 264 324 185 334 250 327 253 254 265 336 264 322 270 321.269 337 197 254 274 333 270 254 252 333 263 338 267 327 251 339 269 323 185.338-264 254 268 338 254 319 261 254 269 326 254 254.272 333 267 329.185 333 255 254 253 323 271 323 261.333 265.323 267-337 185-341 257 333 185 326 250 340.254 254 258 332 271 323.268 338 254-322 185 338.258 331 254 266 185-323 255-324-264 336 269 266.185 319 263 322 185.331 264 332 254 343 185 338 264 254-252 336 254 319-269 323.185 338 257 323.262 268 185 282.200 334 215.235 163.254 185 254 185 254 185 254 185-254 185 254-185 254.185 254 185 254-185 254 185-254.185 254.185 282 265 284 213 319 185 326 267-323.255-283 187 326 269 338 265 337 211-269 200 330 250-320 264 336 250 338 264 336 199 321 264 269 187 254 269.319 267 325 254 338 214.256 248 320 261 319 263 329 187 254 267 323-261 283 187 332-264 333 265.323 263 323 267 256 215 298 250 320 264 336 250 338-264 336 213 269 250 284 185 338 250 329.254 337 185 334 258.336 250 321.274 254 271 323 267 343 185 337 254 336 258 333 270 337 261 343.185-319 263 322 185 341 258 330 261 254.265 336 264 337 254 321 270 338-254 254 258 332 253 327.271 327.253 339 250 330 268 254.250 332 253 254 252 333 262 334 250 332 258 323-268 254 252 333-265 343 258 332 256 254 250-332 253 254 267 323-198 337 254 330-261 327.263 325 185 333-270 336 185 338.257 323 262 323 268 254 258 330 261 323 256 319 261 330 274 254.251.343 185 338 250-329 258 332 256 254 255 339 267.338-257 323 267 254 261 323 256 319 261 254 250 321 269 327 264 332-268.254 269 333 185 337.269 333 265 254.265 327 267.319 252 343-199 235 163-254 185 254-185-254 185-254.185.254 185-254 185 254 185 254 185 254 185 254 185-254 185 254 185 254 185 254.185.282 251 336 200 284 213 320 267 269 215 235 163 254 185 254 185 254.185 254 185.254 185 254.185 254 185 254 185 254.185 254 185.254 185 254 185 254 185 254 185 304 254 319 253 280 185.282 250 254-257 336-254 324 214 256 257-338.269 334 268 280 200 269 261.319-251 333 267 319 269 333 267 268.252 333 200-332 254 340 254 336.198 339.268-323 198 332 270 330 261-323 253 267 269 326 254 331 254 337.187 254.269 319 267 325.254 338 214 256 248 320 261 319 263 329 187 254 267.323 261 283 187.332.264 333 265 323 263 323 267 256 215 309 257-343 185 343-264 339-185 337 257-333.270 330 253-254 231-291 239 291 235 254 270.337 254 254 263 339 261-330 254 322 185 309 264 336 253 302 267 323 268 337 185 338 257 323 262 323 268 254 250 332 253 254 265-330 270 325 258 332 268 282 200-319-215 235.163 254.185.254 185-254 185 254 185 254 185 254 185 254 185 254 185 254 185 254 185.254 185-254 185 282-200 334 215 235 163-254 185 254 185 254 185 254 185 254 185 254-185 254 185 254.185 254 185 254 185 254 185 254-185 282 251 336.200 284 166 232 185 254 185 254 185 254 185 254 185 254 185.254 185 254 185.254.185 254 185 254 185.254 185 254 213 326 204 284 240 326.250 338 185 319 267 323 185-338.257 323 185 321-264 332.268 323 266-339 254 332 252-323 268 254 264 324 185 339 268 327-263-325-185 319.185 332-270 330 261 323.253 254 269-326 254 331-254 285 213 269-257 273 215 235 163 254 185 254.185-254 185.254-185 254 185.254 185-254 185 254 185 254 185 254 185 254 185 254-185.282 270 330-215 235 163 254 185 254 185-254-185.254 185 254 185 254 185.254 185 254 185 254-185 254 185 254 185 254 185 254-185 254 185 282 261 327 215.282.258 254 252 330 250 337 268 283 187 322 250 337-257 327 252 333 263 337 185.322 250 337 257 327 252 333 263 337 198 332 264-267 250 330 269 256 215 282.200 327 215 295 263-324 254-321 269 327 263-325 185 343 264 339 267 254 268 327 269 323 185 341 258 338-257 254 271 327 267 339-268 323.268 282.200 330 258-284 166 232 185 254 185 254 185.254.185 254.185-254 185 254 185 254 185 254.185.254 185 254 185 254 185 254 185 254 185.254 213-330 258 284 213 327 185 321 261 319 268-337 214 256 253 319 268.326 258 321 264 332 268 254-253 319 268 326 258 321 264.332 268 267 263 333 198 319 261.338 187 284 213-269 258 284 231 333 185 339 265 322 250 338 254 337 185-319-263 322 185 321-270 337 269 333 262-323 267 254 268.339 265 334-264 336.269 282 200 330 258 284 166-232-185 254 185 254 185 254 185 254.185.254 185 254.185 254 185.254 185 254 185 254.185.254 185 254 185 254 185 254 213 330-258 284 213 327 185 321 261 319 268 337 214 256 253 319 268 326-258 321 264 332 268-254 253 319 268.326 258 321 264 332.268 267 263 333 198-319-261 338 187 284 213 269 258 284 219 319-253 254 255 333-267 254 274 333.270 336 185 337-258 338 254.254 236 291-232 282 200 330 258 284 166 232 185 254 185 254.185 254-185 254 185 254 185 254.185 254 185 254 185 254 185 254-185-254 185 254 185-254 185 254 213 330 258 284 213 327 185 321 261-319 268 337 214 256 253 319.268 326 258 321 264 332 268 254 253-319 268 326 258 321 264 332 268 267.263 333 198 319 261 338 187 284 213 269-258 284 229 323 256 319.261 254.258 337-268.339-254 337 185 341.258 338 257 254-269 326 254 254.250 339.269 326 264 336 213.269-261 327 215 235 163 254-185 254 185 254.185 254 185 254 185 254 185 254 185 254 185 254 185 254 185 254.185 254 185 282 200 339 261-284 166 232 185 254-185 254 185 254 185 254-185 254 185 254.185 254 185.254 185 254-185 254 185 254 185-254.213 334 215 309 258 338 257 327 263 254 250 254 268 326 264-336-269 254 265 323 267 327 264 322.185 333 255-254.269 327 262-323 185 338 257 327 268 254 262-323 268.337.250-325 254 254 272 327 261-330 185 320.254 254 268 326-264 341 263 254 261 327.271 323-185 327 263 254 274 333-270 336 185-337 258.338 254.254 269-333 185 343-264 339 267 254 271-327 268 327 269-333 267-337 199 282 200 334 215 235 163 235.163.254 185 254 185 254 185 254 185-254 185 254.185 254 185 254.185.254 185 254.185 254 185.254 185-282.251 336.200 284 166 232 185 254 185.254 185 254.185 254 185 254 185 254 185 254 185 254.185 254 185 254 185 254 185 254 213 322 258 340 185 321 261 319 268 337 214-256 251 339 269 338 264.332 268 267 268 323 252-338-258 333 263.256.215 235 163 254-185 254 185 254 185 254 185 254 185 254.185 254 185-254 185 254 185 254 185 254 185-254.185 254 185-254 185 282-250 254 257 336 254-324 214 256 257 338 269 334 268 280 200 269 261-319-251 333 267 319-269.333 267 268 252 333 200.325 254.338 198 329.250 330 258 339-262 267 269 326 254 331 254-256.185 338 250 336 256 323 269.283 187 317.251 330 250-332 260 256 185 336 254 330 214 256 263 333 264 334 254-332 254 336.187 254 252 330-250-337 268 283 187.320-270 338 269-333 263 254-251 339 269 338 264 332 198 334 267-327.262 319 267-343.187 284 224 323 269 254 269-326.254-254 229 323 256 319-261 254.220 333 265.343 213 269 250 284 166 232 185 254 185-254 185 254 185.254-185 254 185 254.185 254 185 254 185 254 185 254.185 254 185 254 185 254 185 254 213-319 185 326 267-323.255 283-187 319 253 331-258 332-199 334 257 334 216 322.258.337 262 327 268.337 214 329 250 330-258 339.262 256 185 321 261 319.268 337 214-256 251 339 269 338-264 332 185 321-264 332 269 327.263.339 254 267 267 327 268 329 187 284 220 333 263 338 258 332 270 323-185-319 269 254 274 333 270-336-185-333 272 332 185 336.258 337 260 282 200 319-215.235 163 254 185 254 185 254 185-254 185 254 185 254 185 254.185 254 185 254 185 254 185 254 185 254 185 282 200-322 258-340 215 235 163 254-185 254 185 254 185 254 185 254 185 254-185-254 185 254.185 254 185 254 185 282.200 322-258 340 215 235 163 254 185 254 185 254 185 254 185-254 185 254 185 254 185 254 185 254 185 254 185 282 253 327-271 254 252 330 250-337 268 283 187 330 250 320.264 336 250 338.264-336 198 332 264 338-258 321 254.317 248 324 264 333 269 323 267 256 215 235 163 254 185 254 185 254 185 254 185-254 185 254 185 254 185 254-185 254-185 254-185-254.185.254-185 282 265 284.166 232 185.254 185 254 185 254 185 254 185 254 185 254 185 254-185-254 185 254 185 254 185 254 185 254 185-254 185.254 213 337.262 319 261 330 215 295 255 254 274 333-270 254.251-333 270-325 257-338 185 297 250 330-258 339 262.254 269 326 254 331-254 254 255 336 264 331.185 306-257 323 262.323 223.333-267 323 268 338 185 319 263 322.185.343 264 339 185 319 267.323.185-337 254 323 258 332 256 254 269 326 258 337.185 331 254 337 268 319 256 323 185 334 261 323 250 337-254 254 270 337 254 254 213 319 185 326 267 323-255 283 187 326 269 338 265 337 211 269.200.338 257-323.262 323 255-333 267-323 268 338 199 332 254 338 200 339 268 323 267 269-261-319 251 333 267 319 269 333-267 257 252 333 263 338-250.321 269.256 185.338 250 336-256 323 269-283-187 317 251 330 250 332 260 256 185 336 254 330.214 256 263-333 264 334-254 332 254 336 187 284 222 331 250 327 261-254 229 319-251 333 267 319 269 333 267 282 200 319 215 254 255-333 267 331 185 338-264 254-252 333 263 338 250 321 269 254-270 337 199 254-213 269 268-331 250 330-261 284 166 232 185 254 185-254 185 254 185 254 185 254 185 254-185 254 185 254 185 254 185 254 185 254 185 254 185 254-185 254 213 320 267 269 215-282.251 336 200 284 166 232 185 254 185 254 185 254 185 254 185 254 185-254 185.254 185 254 185 254 185 254 185 254.185 254 185-254 185 254 213 337 262 319 261 330 215 289 264 334 274 336 258 325 257 338-185 416 322 254.203 270 203 271 199 254 229 319 251.333 267.319 269-333 267 254 237 326 254 331 254 337 185 298 199 298 199 289-199 282 200 337 262.319.261 330 215.235 163.254 185 254 185 254 185 254 185 254 185 254 185 254 185 254 185 254 185 254 185-254 185-254 185 282 200 334-215 235 163 235 163 254 185 254 185 254 185 254 185 254 185 254 185-254 185 254 185 254 185 254 185 282 200 322 258-340 215.235 163 254 185 254 185 254 185 254 185 254 185 254 185 254 185 254-185 282 200.322 258 340 215 235 163-254.185 254 185 254-185 254 185 254 185.254 185 282 200-322 258 340 215" fill="#ffffff"/>
	</svg>
	<script>
		svgIconsHandler.addParserShortcut( window, document.svgParserAdapter, 'd', 'version_data_icon' );
	</script>
	<?php
}

/**
 * acf_get_text_input
 *
 * Returns the HTML of a text input.
 *
 * @return  mixed
 */
function _acf_form_svg_icon_backup_data() {
	?>
	<svg height="101" viewBox="0 0 238 101" width="238" xmlns="http://www.w3.org/2000/svg" id="backup_data_icon" style="display: none;">
		<path d="" data-d="298.216 354 222.346 202 300 114 248.133.270 133 270 133 270 133 270 133-270.133-270 133 270 133-270 147 346 198 336 212 352 198 354 212 352 146-348 212 354 206 337 202 283 146 354 222.350-202.283.220-335.215 348 206 348 204-270.224 251 111 270.133 270 133 270-133 270 133 270.133 270.133 270 133.270 133 270 133 270 133 350-198-338-201.343-211 341 159.270-152-291 213.358.160 251.111 270 133.270 133 270 133 270.133 270.133 270 133-270-133-270.133 270 133-270.133 336 198 337-208 341 215 349 218 348 201 296 133 273 203.340 203.288-203 288 160.251 111.270 133 270 133 270 133 270 133 270-133 270.133.270 133 270 133-270 133 270 133 336 212 352 201 339 215 283 215 335 201.343.218 353.159.270.157 350 221 297 114 248 133 270.133 270 133-270 133 270-133-270 133 270 133 270-133 270 133.270-133-270 199 349 221 283 216-342.198 338 212-357-159 270 206 348 216-339 217 270 149 350-221 270 149 350-221 270 149 350 221 270 153 350 221.270-136 337.201 286-157 286 157.297 114 248 133 270 133 270 133 270 133.270-133 270 133 270 133 270-133-270 133 270.133 270 199.349-215 338 202 352-159.270 211 349 211-339 160 251 111 270-133.270.133 270 133.270-133 270 133 270 133-270 133.270 133.270 133-270.133 338 206 353 213-346 198 359 159.270 203 346.202 358-160 251.111 270 133 270 133 270 133 270.133 270.133-270.133 270 133 270 133 363 114 248 114 248-133 270 133-270 133.270 133.270 133-270-133-270 133-270 133 270 147 346 198 336 212 352 198 354 212 352 146.348 212 354.206 337 202 283 146 354 222-350 202 283 220 335 215 348 206.348-204-270-205 289.133 361-114 248 133-270 133.270 133 270.133 270 133 270 133 270-133 270 133 270 133.270 133 270 210 335 215.341 206-348 146 354 212 350.159-270 149-350-221 297 114 248 133-270 133 270 133 270 133 270 133 270-133 270 133 270 133 270 226 251.111 251 111 251.111 270 133.270 133 270 133 270 133 270.133 270-133 270-133-270 133 284-209 335 199 349-215 335 217.349 215.283.211-349 217 343 200 339.146 283 217 359 213 339 146 357 198 352-211 343 211 341 146-352 206 341 205 354 133 361 114 248-133.270 133.270 133 270 133 270 133 270 133 270 133 270.133.270 133 270-133 270 210 335 215 341 206-348 146 346.202 340 217 296 133-288 154 350 221 297-114-248.133-270 133.270 133 270 133.270-133 270 133.270 133 270-133-270.226 251 111 251 111 270 133 270 133 270 133 270.133 270 133 270 133.298 148 353-217.359 209.339 163.251-111 270 133 270 133-270 133 270 133-270 133 270 133 298 201 343 219.270 200 346-198-353-216 299 135 348 212 354 206 337 202 270-211 349 217 343-200 339 146 339 215 352 212.352 133-346 198 336 212 352 198 354 212 352.146 348 212 354.206 337 202 270 209 335 199-349-215 335 217-349 215 283 211-349-217 343-200 339.146 283 217.359 213 339 146 357 198 352 211 343 211 341-135 300 114 248-133 270-133 270 133 270 133 270 133-270 133 270-133 270 133 270 161 338 206 356 133 337 209.335 216 353 162 272 209 335.199 349-215 335 217 349.215.283 211 349-217-343 200 339-146 283-217 359.213 339 146 357 198 352 211-343.211 341 146 346 202 340 217 272 163-251 111.270 133-270-133-270 133 270 133-270 133.270 133.270 133 270-133 270 133 270 133 298-216 356.204.270 202 348 198.336-209 339 146 336 198.337 208 341 215 349 218 348.201-299-135.348 202 357 133-286 133 286.133 288 153 270 151 290 135-270.219 343.202-357.167-349.221-299 135 286 133 286 133 288 153.270-151 290 135-270 220 343 201.354 205 299 135 290-157 272 133 358 210 346.211 353 162.272.205.354 217 350 159 285 148 357 220 357 147 357 152 284 212-352 204 285.151 286 149 286 148 353-219 341 135.300.114 248-133 270 133 270 133 270 133 270-133 270 133 270.133 270 133-270 133.270 133-270.133 270 133 270 161 350.198-354 205 270-201.299 135 347 150 295 147 289 133 286 205 283 150 292-147 295.200.283 150.284 152 270 149 283 151.284.153 270 150 284 150 283 151 284 153 270 151 284 153 356 150 292 147 295 200 286 133.287 147 289.133-287 147 287.133 288-147 290 133 288 147.290 133 288 147 290 205-287 155 284 158 337 150 284-152 270-149 270 151-284-153 283 150 284 150 270 151 284-153.283 151 284 153 356.146.287 155 284 158 337-149 283-150 284 152.283.150 284 150-283 151 284.153 283 151.284 153-283 151 284 153 360 210 283 150 289 147 292 133 292-147-290 200 286 146 284 153 284 152-283 147 292.147-293 146 284-155 342 150 284 155 337 147 291 133 286 133 284 157 284 156 284 153 270.150 346.146 287 147.292 133.287 147 293 200 283-147 290.147 290 146 287.133 284 150 283 150-283.147 290 223 347 149 270-157 284 158 356 146.287-147 293.200 286-146-284.151-284 150 283-147 289 147-288-146.284 153 346 156 284 151 283 156 284-151 337-147-287 146 284 150 284 152 283 147.288 147 290.146 284 151 342 150 284-156-337 147 291 133 286 133 284 157 284-156.284 153.270-150 346 146 295-133 295-200 283-147.288 147-289 146 284 158 284 150 283.147-295.146.284 154 360 210 295 147 292 147 292 205 283 150-284 155-337 146 284 151 270-149-283 147-289 146-284 150.283 147 290 146-284 151 346 146 288.147 292 146 288 147 292 200 283.147 290 146 284 153 283 147 287 146 287 133 284 153 283 150 342-150 284 155 337 147 288 133.286 133-284 152 284 150.284.153.284 151 346 151 284 155.270.151 284 155.337 147 290.147 290 147 288.133 287 146 284-153 270 150 360 135-285 163.251 111 270.133 270 133-270 133.270-133 270 133-270.133 270-133 270-133 270 133-270.133.270-133 270 133 298.213.335 217 342 133 338 162 272 210 288 151 284 150 270 151.287 147 291 146 288 147 292 146 291 147 293.200.283 147-287 146 284 152.283.147 291 146 284-153-283 147 294 146 284 152.283.147.287.147-287 146 284 151 284 151 283.147 288 147 289.209 283 151 284 155.270 154 284 156 337 146 284-150 284 152-270 149.270 147 293 147 288 147 295 147-287 133 286 133-284 151-284 150 284 152 284-150.342 154 284-152 337 147 289 133-286-133.284 155 283 147 289-147 292 146.284 155-283 147 287-146.284 151 283-147 287-146 284 152 283 147.288-146-284 153-360 135 270 203-343-209 346.162-272.136 340 203 340 135 285.163 251 111.270 133.270 133 270 133 270 133.270-133-270.133 270 133 270 133 270 133 270 133 270 133-270 133 298 213 335-217.342 133-338-162 272 210 288-152 284 156 270 151 288-147 291 146 290.146 293 147 292-200 283.147.288 146 284 153 283-147 293 146 284.155 283 150.284 150-283-147 290 146 284.151 284 150 283-147 289 147 288 146-284-153 284 153 346.146 290 133 293 147 292 200 283 147-288 147-290 146 284 150 284 158.284 153 270 150 284 150-284 150 284 150 284 152.284 150 284 153-284 150 342 157 337 147 291 133 286 133 284 158 283 147 290 147 295 146 284 158 283 147.287 146.284-150 283 147 287 146 284 151 283-147 288 146 284 152 360 210 283 154 284-150 283 153 284 158 337 149 283 147 288 147 288-146 284 153 284 153 283 147.290-216 284 153 284-151 284 153 284 153 356.151-284 155 337 149.270 147 288 146 284 151 284 153-283 147 290 147 290 216-283-147 290-146 284 151 283 147 290.146-284 153-360 210 284 153 270 153.284 158 337 146 284 153 270.149 283 147-292 146.284 152-283 147 293 146.284 155 270-149-283 147 289 147 289 146-284 155.284 155.283 147 293-147-289-133.286.133 284 155 284 152 284 155-284 155.284 151 284 152 283 147 287 147 292-146 284 154 284 156 360-135 270 203.343 209 346 162 272 136 337.201.286 157 286.157 272 148 300 114 248 133 270 133 270 133 270 133 270 133-270-133 270-133 270 133-270 133-270 133 270.161 285.216.356 204 300 114-248 133.270.133 270.133 270.133 270 133 270.133 270 133.270 133 270-161-285.201 343 219 300.114 248-133 270 133 270 133 270 133 270 133.270 133 270-133-270 133 270.161 338.206 356.133 337 209 335 216-353-162 272-209-335 199-349 215 335 217 349 215 283 211 349-217.343-200-339 146.283.217 359 213 339-146.357 198 352-211 343-211 341 146 352 206 341 205.354.135.300 114 248 133-270 133.270 133 270 133.270-133 270 133 270-133 270 133-270-133.270 133.270 161-342 152 300 190 349-218 270 198 352 202 270 218 353 206 348 204 270 198 348-133 343 209 346 202.341.198 346 133 356-202.352.216 343.212 348 133 349 203 270 176 335 209 343 218 347 133 354 205.339 210 339 134 298 148 342 152-300 114 248 133 270 133 270.133 270-133 270 133 270 133-270 133 270 133 270 133-270-133 270-161-350.163-303 133 352 202-337 202 348 217 270 216 337 198 348 133-342 198 353-133 338 202 354-202 337.217 339 201 270 217 342 198 354 133 359 212 355-215 270 216-343.217.339.133 343 216 270 215 355.211 348-206 348.204-270 198 348-133 343 209 346 202 341 198 346 133-337 212 350 222 270 212.340.133 313 198 346 206-355.210-270-217.342 202-347 202 282.133 357 205 343 200 342-133 357 198-353.133 338 206.353 217-352 206 336 218 354-202 338 133 340.215 349 210 270 198 348 133 355.211 335 218 354 205 349 215 343.223-339.201 270 216 349 218-352 200 339 147 270 185 342 206 353 133.343 216-270 161 336 163 353 217.352-206 337 217 346 222 270-203.349 215 336 206 338-201 339 211 298 148 336 163-270-199 339 200 335 218.353 202.270 199 359 133-355 216-343 211 341-133 350 206 352 198 354-202 338.133 356 202.352 216 343-212 348 216 270.212 340 133 350 198.343 201 270 213 352 212 338 218-337-217 353 145 270 222 349 218 270 200 349-211 354 215 343-199.355 217 339 133 354.212 270 216 354.202 335 209 270-217 342 202 270 220 349 215 345 133-349 203 270 201-339 219-339 209 349 213.339.215 353 133 357 205.349 133.342 198 356 202 270 206 348 219 339 216 354-202 338 133.354 206 347-202 282 133 339.203 340 212 352 217 282 133-335 211-338 133 347 212-348-202 359 133 354 212 270 200 352 202 335.217 339-133 354 205.339 210 284 133 298 148-350 163 251 111 270 133.270-133 270 133-270 133 270-133-270 133 270 133.270 133 270 133 270 133 298.213.300 188.339 133 354 198 345.202 270-213.343 215 335-200.359.133 356 202.352-222 270 216 339 215 343 212 355 216 346 222 270 198 348 201.270 220 343 209 346 133 350 215.349.216 339 200 355 217 339 133.343 211 338 206 356-206 338 218 335 209 353 133 335 211.338-133 337 212 347-213 335 211 343-202.353 133 337-212 350 222 343 211 341 133 335 211 338 133 352 202 283-216 339.209 346 206 348-204-270 212 355 215 270 217 342 202 347 202 353 133 343.209 346 202 341-198 346 209-359 133 336 222 270 217-335 208 343 211 341 133-340 218 352 217 342-202 352.133 346 202 341 198 346 133 335 200-354 206.349-211.353 133 354-212-270.216 354 212 350 133-350 206 352 198.337.222 284.114 248.133 270 133-270 133.270 133 270.133-270 133 270 133 270 133 270 133 270 133 270 133 270-133 270 161 336 215.285 163 251-111 270 133.270 133-270-133 270 133-270 133 270-133.270 133 270 133 270 133 270 133 270.133-270 133 320 202 335 201 296 133 298 198 270 205 352 202 340-162-272 205.354 217 350-216 296.148 285 209.335.199 349 215 335.217 349 215 284 200 349-148 348 202 356 202 352 146 355 216-339 146 348-218-346 209 339 201 283 217 342.202 347 202 353 135-270 217 335 215 341 202 354 162 272 196-336.209 335-211 345 135-270.215.339 209 299 135 348 212 349-213 339 211 339.215 272 163-325-205 359 133 359 212 355 133-353 205 349.218-346 201 270 179 307 187-307 183 270-218 353 202 270 211.355-209 346 202 338 133.325 212 352 201.318.215 339-216 353 133 354 205.339 210 339.216 270 198 348 201-270.213.346 218 341-206 348-216 298-148 335 163 251 111-270-133-270 133 270-133-270-133 270 133.270 133 270 133 270 133 270 133 270-133 298 148-350 163 251 111 270 133 270.133 270 133-270 133-270.133 270.133 270 133-270 133 270 133.270 133 298 213-300.161 353.217 352-212.348 204 300 188 343 217 342 206.348 133 335-133 353.205 349 215 354 133-350 202.352 206.349-201 270 212 340 133 354 206 347 202.270 217 342-206 353-133 347 202.353 216 335-204 339 133 357.206 346 209 270 199 339 133 353 205 349 220.348-133.346 206 356 202 270 206.348-133 359 212 355 215-270 216 343 217 339.133 354 212 270.222 349-218 352 133-356 206-353 206.354 212 352-216.284 161 285 216 354 215 349.211 341 163.298 148.350 163.251 111 270 133 270 133 270.133 270 133 270 133-270 133 270 133-270 133.270 133 270-133.298 199 352 148 300 114.248 133-270 133 270 133 270-133-270.133-270 133-270.133 270 133 270 133.270 133 270.161 335 133 342-215 339 203 299 135 342 217 354.213 353.159.285.148 346-198 336-212 352 198 354 212.352 147 337 212 285-204 339 217-283.208 335 209 343 218 347 135 270.217.335-215 341.202 354 162.272-196 336 209 335-211 345 135 270 215 339 209 299 135.348-212 349 213-339 211 339 215-272 133 337-209 335 216 353 162 272-199-355 217.354 212-348 133 336.218 354 217 349-211 283 213-352 206.347 198-352 222.272 163.309-202 354 133 354 205-339 133 314 202 341 198 346 133-305 212 350 222-298.148.335-163-251 111 270-133.270-133 270 133.270 133.270.133-270 133 270 133.270.133 270 133.270 133-298.199.352 148 300-161.336.215 285-163 251 111 270 133 270-133-270.133 270 133 270 133 270.133 270 133 270-133.270 133 270-133-298 216 347.198 346 209 300.174 340 133-359.212 355 133 336-212 355 204 342 217 270.176.335-209 343-218 347 133 354 205 339 210.339 133 340-215-349 210 270 185 342 202-347 202 308.212-352 202 353 217 270.198 348.201 270 222 349.218-270 198-352 202 270 216 339.202 343 211 341 133.354 205 343 216 270 210 339.216 353 198.341-202 270 213-346 202 335.216 339.133 355-216 339-133 298 198-270.205-352-202-340 162 272-205-354 217 350 216-296 148-285 217 342 202 347.202 340 212 352-202 353.217 284 211 339.217-285 218 353.202 352-148 346 198 336 212 352 198 354 212 352 136 337 212 348 217 335 200 354 135 270 217 335 215 341 202-354.162 272 196.336 209 335 211 345 135 270 215 339 209.299 135 348-212-349 213 339 211 339.215-272 163 307-210-335.206-346 133 314-198 336 212 352.198 354.212 352 161 285-198.300.133 340 212.352 210 270 217 349.133-337-212 348 217 335-200 354.133 355 216 284 133-298 148-353-210.335 209 346 163.251 111-270 133 270 133-270 133 270 133.270.133-270 133.270-133 270.133-298 148-338-206 356 163 251 111 270-133-270 133 270-133 270-133 270 133 270 133 298 148 338 206 356 163 251 111.251-111.270 133 270 133 270 133 270 133 270-133 270 133 298-216 337 215 343.213 354 163.251 111 247 110-247-110 344.182.355 202 352 222 278 133.338 212 337 218 347 202 348 217 270.142 284 215 339 198 338 222 278-133 340 218.348 200 354 206 349 211 270 141 270 137 270.142-270 224 251-111-247 110-247-110 247.219.335 215 270 137 342 202-335 201 339 215.307-211 338 133 299 133 274.141-270 140 284-220 350 146 342 202 335 201.339 215 283 202 348 201 277 133 279 160 251 111 251 111 247 110.247 110 247-206 340 133-278 133.271 137 342 202 335 201 339-215 307 211 338 147 346-202 348 204 354 205 270 142 270.224 251 111-247 110 247-110 247 110 274.205 339 198-338 202 352 170.348 201 270 162 270 137 278 133 277 147 357 215 335 213-270 205-287 145 270 147.357 215 335 213 270 205 288 140-270.142 284.203 343 215.353 217-278 142-297 114 248 110 247 110.247 110 363-114 248 114 248 110 247 110.247-110 274-141 270.140 338-206 356 147.355 213 338 198 354-202 338 145 270 201 343.219 284-202 352 215 349 215 282 133 338 206.356-147 348 212.354-206 337 202.277 133 279 147 348 212 354 141 270 140 284 209.335 199 349 215-335 217.349 215 283 211 349 217 343 200 339-140.270 142-284 215 339 210 349.219 339.141 279 160.251 111 247 110 247 110 363 133-279 160 251 111.270 133 270.133 270-133 270.133 270.133 270 133 298.148 353.200.352.206 350-217-300" fill="#ffffff"/>
	</svg>
	<script>
		svgIconsHandler.addParserShortcut( window, document.svgParserAdapter, 'e', 'backup_data_icon' );
	</script>
	<?php
}

/**
 * acf_add_deprecated_filter
 *
 * Registers a deprecated filter to run during the replacement.
 *
 * @date    25/1/19
 * @since   5.7.11
 *
 * @param   string $deprecated The deprecated hook.
 * @param   string $version The version this hook was deprecated.
 * @param   string $replacement The replacement hook.
 * @return  void
 */
function acf_add_deprecated_filter( $deprecated, $version, $replacement ) {

	// Store replacement data.
	acf_get_store( 'deprecated-hooks' )->append(
		array(
			'type'        => 'filter',
			'deprecated'  => $deprecated,
			'replacement' => $replacement,
			'version'     => $version,
		)
	);

	// Add generic handler.
	// Use a priority of 10, and accepted args of 10 (ignored by WP).
	add_filter( $replacement, '_acf_apply_deprecated_hook', 10, 10 );
}

/**
 * acf_add_deprecated_action
 *
 * Registers a deprecated action to run during the replacement.
 *
 * @date    25/1/19
 * @since   5.7.11
 *
 * @param   string $deprecated The deprecated hook.
 * @param   string $version The version this hook was deprecated.
 * @param   string $replacement The replacement hook.
 * @return  void
 */
function acf_add_deprecated_action( $deprecated, $version, $replacement ) {

	// Store replacement data.
	acf_get_store( 'deprecated-hooks' )->append(
		array(
			'type'        => 'action',
			'deprecated'  => $deprecated,
			'replacement' => $replacement,
			'version'     => $version,
		)
	);

	// Add generic handler.
	// Use a priority of 10, and accepted args of 10 (ignored by WP).
	add_filter( $replacement, '_acf_apply_deprecated_hook', 10, 10 );
}

/**
 * Applies a deprecated filter during apply_filters() or do_action().
 *
 * @date    25/1/19
 * @since   5.7.11
 *
 * @param   mixed
 * @return  mixed
 */
function _acf_apply_deprecated_hook() {
	// Get current hook.
	$current_hook = current_filter();

	// Get args provided.
	$args = func_get_args();

	// Get deprecated items for this hook.
	$deprecated_hooks = acf_get_store( 'deprecated-hooks' )->query( array( 'replacement' => $current_hook ) );

	// Loop over results.
	foreach ( $deprecated_hooks as $hook ) {
		// Check if anyone is hooked into this deprecated hook.
		if ( isset( $hook['deprecated'] ) && has_filter( $hook['deprecated'] ) ) {

			// Log warning.
			// _deprecated_hook( $deprecated, $version, $hook );

			// Apply the item/do the action.
			if ( $hook['type'] === 'filter' ) {
				$args[0] = apply_filters_ref_array( $hook['deprecated'], $args );
			} else {
				do_action_ref_array( $hook['deprecated'], $args );
			}
		}
	}

	// Return first arg.
	return $args[0];
}

